#!/usr/bin/env python3
"""
slice_flip_counts.py
====================

Copy per-L flip-count arrays from orig/flip_counts or from a simulation
results directory (data/results/vol4_loop_fluctuation_sim) into data/flip_counts.
No synthetic data.

Usage:
    python scripts/slice_flip_counts.py --L_values 6 8 10 12 16 20 --out data/flip_counts/ [--indir data/results/vol4_loop_fluctuation_sim]
"""

import argparse
import os
import shutil
import sys

def main():
    parser = argparse.ArgumentParser(
        description="Copy per-L flip_counts into data/flip_counts"
    )
    parser.add_argument(
        "--L_values", required=True, nargs="+", type=int,
        help="Lattice sizes to copy, e.g. 6 8 10 12 16 20"
    )
    parser.add_argument(
        "--out", required=True,
        help="Output directory for per-L flip_counts (e.g. data/flip_counts/)"
    )
    parser.add_argument(
        "--indir", required=False,
        help="If provided, look under this folder for subdirs L{L}/flip_counts.npy"
    )
    args = parser.parse_args()

    os.makedirs(args.out, exist_ok=True)

    for L in args.L_values:
        if args.indir:
            # e.g. data/results/vol4_loop_fluctuation_sim/L6/flip_counts.npy
            src = os.path.join(args.indir, f"L{L}", "flip_counts.npy")
        else:
            src = f"orig/flip_counts/flip_counts_L{L}.npy"

        if not os.path.isfile(src):
            print(f"Error: Missing flip-count file: {src}", file=sys.stderr)
            sys.exit(1)

        dst = os.path.join(args.out, f"flip_counts_L{L}.npy")
        shutil.copy2(src, dst)
        print(f"Copied {src} -> {dst}")

if __name__ == "__main__":
    main()
